<?php
namespace Barn2\Plugin\Document_Library_Pro\Util;

use Barn2\Plugin\Document_Library_Pro\Dependencies\enshrined\svgSanitize\Sanitizer;

defined( 'ABSPATH' ) || exit;

/**
 * Handles File Icon SVGs
 *
 * @package   Barn2\document-library-pro
 * @author    Barn2 Plugins <support@barn2.com>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
final class SVG_Icon {

	/**
	 * Renders the SVG Icon.
	 *
	 * @param string $key
	 * @param array $classes
	 */
	public static function render( $key, $classes = [], $fill_color = '' ) {
		$svg = self::get( $key, $classes, $fill_color );

		if ( $svg === false ) {
			return;
		}

		echo $svg; // phpcs:ignore
	}

	/**
	 * Get the SVG Icon.
	 *
	 * @param string $key
	 * @param array $classes
	 * @return string|false
	 */
	public static function get( $key, $classes = [], $fill_color = '' ) {
		$icons = apply_filters(
			'document_library_pro_icon_svg',
			[
				'www'          => '<svg xmlns="http://www.w3.org/2000/svg" x="0" y="0" viewBox="0 0 511 511" xml:space="preserve"><style>.prefix__st2{fill:#9777a8}.prefix__st3{fill:#fff}</style><path d="M337.5 0H72.7c-7.4 0-13.4 6-13.4 17.6v484.3c0 3.2 6 9.1 13.4 9.1h365.7c7.4 0 13.3-6 13.3-9.1V118.4c0-6.4-.8-8.4-2.3-9.9L343.2 2.4C341.7.8 339.6 0 337.5 0zm0 0" fill="#e9e9e0"/><path d="M342.2 1.4v108.1h108.1L342.2 1.4zm0 0" fill="#d9d7ca"/><path class="prefix__st2" d="M438.3 511H72.7c-7.4 0-13.4-6-13.4-13.4V355.9h392.3v141.8c.1 7.3-5.9 13.3-13.3 13.3zm0 0"/><g id="prefix__www"><path class="prefix__st3" d="M102.2 483.3l-27.1-92.6h17l17.5 67.2 16.4-67.2h11.3l16.4 67.1 17.3-67.1h16.2l-27.1 92.6h-13.4l-15.8-63.5-15.9 63.5h-12.8zM224.4 483.3l-27.1-92.6h17l17.5 67.2 16.4-67.2h11.3l16.4 67.1 17.3-67.1h16.2l-27.1 92.6H269l-15.9-63.5-15.9 63.5h-12.8zM346.7 483.3l-27.1-92.6h17l17.5 67.2 16.5-67.2h11.3l16.4 67.1 17.3-67.1h16.2l-27.1 92.6h-13.4l-15.8-63.5-15.9 63.5h-12.9z"/></g><path class="prefix__st2" d="M235.2 249l-21.7 21.7c-9 9-23.5 9-32.5 0s-9-23.5 0-32.5l43.3-43.3c9-9 23.5-9 32.5 0 3 3 7.8 3 10.8 0 3-3 3-7.8 0-10.8-15-15-39.2-15-54.1 0l-43.3 43.3c-15 15-15 39.2 0 54.1 15 15 39.2 15 54.2 0l21.7-21.7c3-3 3-7.8 0-10.8-3-3-7.9-3-10.9 0z"/><path class="prefix__st2" d="M331.6 120.1c-15-15-39.2-15-54.2 0l-26 26c-3 3-3 7.8 0 10.8 3 3 7.8 3 10.8 0l26-26c9-9 23.5-9 32.5 0s9 23.5 0 32.5L273.1 211c-9 9-23.5 9-32.5 0-3-3-7.8-3-10.8 0-3 3-3 7.8 0 10.8 15 15 39.2 15 54.1 0l47.6-47.6c15.1-14.9 15.1-39.2.1-54.1z"/></svg>',
				'default'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><circle cx="25.5" cy="25" r="10" fill="#fff"/><path d="M25.5 36c-6.065 0-11-4.935-11-11s4.935-11 11-11 11 4.935 11 11-4.935 11-11 11zm0-20c-4.963 0-9 4.037-9 9s4.037 9 9 9 9-4.037 9-9-4.037-9-9-9z" fill="#556080"/><path d="M42.5 45a.997.997 0 01-.707-.293l-11-11a.999.999 0 111.414-1.414l11 11A.999.999 0 0142.5 45z" fill="#556080"/><path d="M25.5 22h-6a1 1 0 110-2h6a1 1 0 110 2zM30.5 22h-2a1 1 0 110-2h2a1 1 0 110 2zM24.5 30h-2a1 1 0 110-2h2a1 1 0 110 2zM30.5 30h-2a1 1 0 110-2h2a1 1 0 110 2zM23.5 26h-4a1 1 0 110-2h4a1 1 0 110 2zM32.5 26h-5a1 1 0 110-2h5a1 1 0 110 2z" fill="#d1d4d1"/></svg>',
				'exe'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#9777a8"/><g fill="#fff"><path d="M17.082 44.168v3.172h4.211v1.121h-4.211v3.295h4.635V53h-6.303V42.924h6.303v1.244h-4.635zM28.58 48.105L31.137 53h-1.9l-1.6-3.801H27.5L25.777 53h-1.9l2.557-4.895-2.721-5.182h1.873l1.777 4.102h.137l1.928-4.102h1.873l-2.721 5.182zM35.266 44.168v3.172h4.211v1.121h-4.211v3.295H39.9V53h-6.303V42.924H39.9v1.244h-4.634z"/></g><path d="M33.5 32a1.001 1.001 0 01-.955-1.299l5-16a1 1 0 011.909.597l-5 16A1 1 0 0133.5 32z" fill="#9777a8"/><circle cx="29" cy="19.5" r="1.5" fill="#9777a8"/><circle cx="29" cy="26.5" r="1.5" fill="#9777a8"/><path d="M23.5 30h-2c-3.86 0-7-3.14-7-7s3.14-7 7-7h2a1 1 0 010 2h-2c-2.757 0-5 2.243-5 5s2.243 5 5 5h2a1 1 0 010 2z" fill="#9777a8"/></svg>',
				'archive'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#556080"/><g fill="#fff"><path d="M25.266 42.924v1.326l-4.799 7.205-.273.219h5.072V53h-6.699v-1.326l4.799-7.205.287-.219h-5.086v-1.326h6.699zM29.271 53h-1.668V42.924h1.668V53zM33.414 53h-1.641V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232h-1.217V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23z"/></g><g fill="#c8bdb8"><path d="M28.5 24v-2h2v-2h-2v-2h2v-2h-2v-2h2v-2h-2v-2h2V8h-2V6h-2v2h-2v2h2v2h-2v2h2v2h-2v2h2v2h-2v2h2v2h-4v5c0 2.757 2.243 5 5 5s5-2.243 5-5v-5h-4zm2 5c0 1.654-1.346 3-3 3s-3-1.346-3-3v-3h6v3z"/><path d="M26.5 30h2a1 1 0 100-2h-2a1 1 0 100 2z"/></g></svg>',
				// doc files
				'spreadsheet'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#91cda0"/><g fill="#fff"><path d="M20.379 48.105L22.936 53h-1.9l-1.6-3.801h-.137L17.576 53h-1.9l2.557-4.895-2.721-5.182h1.873l1.777 4.102h.137l1.928-4.102H23.1l-2.721 5.182zM27.037 42.924v8.832h4.635V53h-6.303V42.924h1.668zM39.041 50.238c0 .364-.075.718-.226 1.06s-.362.642-.635.902-.611.467-1.012.622a3.771 3.771 0 01-1.367.232c-.219 0-.444-.012-.677-.034s-.467-.062-.704-.116a3.686 3.686 0 01-.677-.226 2.17 2.17 0 01-.554-.349l.287-1.176c.127.073.289.144.485.212s.398.132.608.191c.209.06.419.107.629.144.209.036.405.055.588.055.556 0 .982-.13 1.278-.39.296-.26.444-.645.444-1.155 0-.31-.105-.574-.314-.793-.21-.219-.472-.417-.786-.595s-.654-.355-1.019-.533a6.173 6.173 0 01-1.025-.629 3.281 3.281 0 01-.793-.854c-.21-.328-.314-.738-.314-1.23 0-.446.082-.843.246-1.189s.385-.641.663-.882.602-.426.971-.554.759-.191 1.169-.191c.419 0 .843.039 1.271.116.428.077.774.203 1.039.376a13.31 13.31 0 01-.191.39l-.205.396c-.064.123-.119.226-.164.308a1.263 1.263 0 01-.082.137c-.055-.027-.116-.063-.185-.109s-.167-.091-.294-.137a2.097 2.097 0 00-.506-.096 4.781 4.781 0 00-.807.014c-.183.019-.355.07-.52.157s-.31.193-.438.321a1.464 1.464 0 00-.301.431 1.1 1.1 0 00-.109.458c0 .364.104.658.314.882.209.224.469.419.779.588.31.169.647.333 1.012.492.364.159.704.354 1.019.581s.576.513.786.854c.212.342.317.781.317 1.319z"/></g><path d="M23.5 16v-4h-12v22h33V16h-21zm-10-2h8v2h-8v-2zm0 4h8v2h-8v-2zm0 4h8v2h-8v-2zm0 4h8v2h-8v-2zm8 6h-8v-2h8v2zm21 0h-19v-2h19v2zm0-4h-19v-2h19v2zm0-4h-19v-2h19v2zm-19-4v-2h19v2h-19z" fill="#c8bdb8"/></svg>',
				'word'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M18.5 13h-6a1 1 0 010-2h6a1 1 0 010 2zM21.5 18h-9a1 1 0 010-2h9a1 1 0 010 2zM25.5 18c-.26 0-.52-.11-.71-.29-.18-.19-.29-.45-.29-.71 0-.26.11-.52.29-.71.37-.37 1.05-.37 1.42 0 .18.19.29.45.29.71 0 .26-.11.52-.29.71-.19.18-.45.29-.71.29zM37.5 18h-8a1 1 0 010-2h8a1 1 0 010 2zM12.5 33c-.26 0-.52-.11-.71-.29-.18-.19-.29-.45-.29-.71 0-.26.11-.52.29-.71.37-.37 1.05-.37 1.42 0 .18.19.29.44.29.71 0 .26-.11.52-.29.71-.19.18-.45.29-.71.29zM24.5 33h-8a1 1 0 010-2h8a1 1 0 010 2zM43.5 18h-2a1 1 0 010-2h2a1 1 0 010 2zM34.5 23h-22a1 1 0 010-2h22a1 1 0 010 2zM43.5 23h-6a1 1 0 010-2h6a1 1 0 010 2zM16.5 28h-4a1 1 0 010-2h4a1 1 0 010 2zM30.5 28h-10a1 1 0 010-2h10a1 1 0 010 2zM43.5 28h-9a1 1 0 010-2h9a1 1 0 010 2z" fill="#8697cb"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#0096e6"/><g fill="#fff"><path d="M23.5 47.682c0 .829-.089 1.538-.267 2.126s-.403 1.08-.677 1.477-.581.709-.923.937-.672.398-.991.513a4.094 4.094 0 01-.875.219c-.264.03-.46.046-.587.046h-3.814V42.924H18.4c.848 0 1.593.135 2.235.403s1.176.627 1.6 1.073.74.955.95 1.524c.21.57.315 1.156.315 1.758zm-4.867 4.115c1.112 0 1.914-.355 2.406-1.066s.738-1.741.738-3.09c0-.419-.05-.834-.15-1.244-.101-.41-.294-.781-.581-1.114s-.677-.602-1.169-.807-1.13-.308-1.914-.308h-.957v7.629h1.627zM33.475 47.914c0 .848-.107 1.595-.321 2.242-.214.647-.511 1.185-.889 1.613-.378.429-.82.752-1.326.971s-1.06.328-1.661.328-1.155-.109-1.661-.328-.948-.542-1.326-.971c-.378-.429-.675-.966-.889-1.613-.214-.647-.321-1.395-.321-2.242s.107-1.593.321-2.235c.214-.643.51-1.178.889-1.606.378-.429.82-.754 1.326-.978s1.06-.335 1.661-.335 1.155.111 1.661.335.948.549 1.326.978c.378.429.674.964.889 1.606.213.642.321 1.387.321 2.235zm-4.239 3.815c.337 0 .658-.066.964-.198.305-.132.579-.349.82-.649.241-.301.431-.695.567-1.183s.209-1.082.219-1.784c-.009-.684-.08-1.265-.212-1.743-.132-.479-.314-.873-.547-1.183s-.497-.533-.793-.67a2.203 2.203 0 00-.937-.205c-.337 0-.659.063-.964.191a2.065 2.065 0 00-.82.649c-.242.306-.431.699-.567 1.183s-.21 1.075-.219 1.777c.009.684.08 1.267.212 1.75.132.483.314.877.547 1.183s.497.528.793.67c.297.141.609.212.937.212zM42.607 51.975a3.73 3.73 0 01-1.271.82 4.212 4.212 0 01-1.531.273c-.602 0-1.155-.109-1.661-.328s-.948-.542-1.326-.971c-.378-.429-.675-.966-.889-1.613-.214-.647-.321-1.395-.321-2.242s.107-1.593.321-2.235c.214-.643.51-1.178.889-1.606a3.78 3.78 0 011.333-.978 4.069 4.069 0 011.654-.335c.547 0 1.057.091 1.531.273.474.183.897.456 1.271.82l-1.135 1.012a2.072 2.072 0 00-1.627-.752c-.337 0-.659.063-.964.191a2.065 2.065 0 00-.82.649c-.242.306-.431.699-.567 1.183s-.21 1.075-.219 1.777c.009.684.08 1.267.212 1.75.132.483.314.877.547 1.183s.497.528.793.67c.296.142.608.212.937.212s.636-.06.923-.178.549-.31.786-.574l1.134.999z"/></g></svg>',
				'presentation' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#f6712e"/><g fill="#fff"><path d="M17.581 53H15.94V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232H17.58V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23zM25.853 53h-1.641V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232h-1.217V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23zM39.606 42.924v1.121h-3.008V53h-1.654v-8.955h-3.008v-1.121h7.67z"/></g><path d="M39.5 30h-24V14h24v16zm-22-2h20V16h-20v12z" fill="#c8bdb8"/><path d="M20.499 35a1 1 0 01-.857-1.515l3-5a1 1 0 111.715 1.029l-3 5a.997.997 0 01-.858.486zM34.501 35a.998.998 0 01-.858-.485l-3-5a1 1 0 111.715-1.029l3 5A1 1 0 0134.501 35zM27.5 16a1 1 0 01-1-1v-3a1 1 0 112 0v3a1 1 0 01-1 1z" fill="#c8bdb8"/><path fill="#d3ccc9" d="M17.5 16h20v12h-20z"/></svg>',
				'pdf'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M19.514 33.324c-.348 0-.682-.113-.967-.326-1.041-.781-1.181-1.65-1.115-2.242.182-1.628 2.195-3.332 5.985-5.068 1.504-3.296 2.935-7.357 3.788-10.75-.998-2.172-1.968-4.99-1.261-6.643.248-.579.557-1.023 1.134-1.215a4.91 4.91 0 011.016-.172c.504 0 .947.649 1.261 1.049.295.376.964 1.173-.373 6.802 1.348 2.784 3.258 5.62 5.088 7.562 1.311-.237 2.439-.358 3.358-.358 1.566 0 2.515.365 2.902 1.117.32.622.189 1.349-.39 2.16-.557.779-1.325 1.191-2.22 1.191-1.216 0-2.632-.768-4.211-2.285-2.837.593-6.15 1.651-8.828 2.822-.836 1.774-1.637 3.203-2.383 4.251-1.025 1.435-1.909 2.105-2.784 2.105zm2.662-5.126c-2.137 1.201-3.008 2.188-3.071 2.744-.01.092-.037.334.431.692.149-.047 1.019-.444 2.64-3.436zm13.637-4.442c.815.627 1.014.944 1.547.944.234 0 .901-.01 1.21-.441.149-.209.207-.343.23-.415-.123-.065-.286-.197-1.175-.197-.505.001-1.14.023-1.812.109zm-7.47-6.582a71.291 71.291 0 01-2.674 7.564 49.966 49.966 0 016.496-2.02c-1.35-1.568-2.699-3.526-3.822-5.544zm-.607-8.462c-.098.033-1.33 1.757.096 3.216.949-2.115-.053-3.23-.096-3.216zM48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#cc4b4c"/><g fill="#fff"><path d="M17.385 53h-1.641V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232h-1.217V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335.164-.155.296-.371.396-.649.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23zM32.219 47.682c0 .829-.089 1.538-.267 2.126s-.403 1.08-.677 1.477-.581.709-.923.937-.672.398-.991.513a4.094 4.094 0 01-.875.219c-.264.03-.46.046-.588.046h-3.814V42.924h3.035c.848 0 1.593.135 2.235.403s1.176.627 1.6 1.073.74.955.95 1.524c.21.57.315 1.156.315 1.758zm-4.867 4.115c1.112 0 1.914-.355 2.406-1.066s.738-1.741.738-3.09c0-.419-.05-.834-.15-1.244-.101-.41-.294-.781-.581-1.114s-.677-.602-1.169-.807-1.13-.308-1.914-.308h-.957v7.629h1.627zM36.266 44.168v3.172h4.211v1.121h-4.211V53h-1.668V42.924H40.9v1.244h-4.634z"/></g></svg>',
				// audio files
				'mp3'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#7d6599"/><path d="M35.67 14.986c-.567-.796-1.3-1.543-2.308-2.351-3.914-3.131-4.757-6.277-4.862-6.738V5a1 1 0 10-2 0v18.412h-3.706c-3.882 0-6.294 1.961-6.294 5.117 0 3.466 2.24 5.706 5.706 5.706a6.3 6.3 0 006.294-6.294V16.468l.298.243c.34.336.861.72 1.521 1.205 2.318 1.709 6.2 4.567 5.224 7.793a1 1 0 101.914.581c1.627-5.375-.422-9.39-1.787-11.304zM26.5 27.941a4.299 4.299 0 01-4.294 4.294c-2.355 0-3.706-1.351-3.706-3.706 0-2.576 2.335-3.117 4.294-3.117H26.5v2.529zm5.005-11.633c-.571-.422-1.065-.785-1.371-1.081l-1.634-1.34v-3.473c.827 1.174 1.987 2.483 3.612 3.783.858.688 1.472 1.308 1.929 1.95a10.832 10.832 0 011.788 3.978c-1.327-1.61-3.084-2.904-4.324-3.817z" fill="#c8bdb8"/><g fill="#fff"><path d="M22.4 42.924h1.668V53H22.4v-6.932l-2.256 5.605h-1.449l-2.27-5.605V53h-1.668V42.924h1.668l2.994 6.891 2.981-6.891zM28.211 53H26.57V42.924h2.898c.429 0 .853.068 1.271.205s.795.342 1.128.615.602.604.807.991.308.822.308 1.306c0 .511-.087.973-.26 1.388s-.415.764-.725 1.046-.684.501-1.121.656-.921.232-1.449.232H28.21V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103s.376-.18.54-.335.296-.371.396-.649.15-.622.15-1.032c0-.164-.022-.354-.068-.567s-.139-.419-.28-.615-.34-.36-.595-.492-.593-.198-1.012-.198h-1.23zM38.615 47.695a2.2 2.2 0 011.271.67c.191.2.351.444.479.731s.191.617.191.991c0 .456-.066.85-.198 1.183s-.299.608-.499.827-.426.395-.677.526-.494.23-.731.294-.453.104-.649.123-.349.027-.458.027c-.766 0-1.369-.053-1.812-.157s-.75-.212-.923-.321l.369-1.176c.082.046.159.096.232.15s.178.107.314.157.328.091.574.123.583.048 1.012.048c.629 0 1.099-.171 1.408-.513s.465-.772.465-1.292c0-.492-.142-.907-.424-1.244s-.697-.506-1.244-.506H35.92h.014l-.014-1.107h.93c.2 0 .403-.005.608-.014s.398-.06.581-.15.333-.246.451-.465.178-.533.178-.943c0-.164-.014-.337-.041-.52s-.103-.351-.226-.506-.303-.28-.54-.376-.565-.144-.984-.144-.754.039-1.005.116-.439.139-.567.185l-.479-1.23c.219-.063.49-.132.813-.205s.791-.109 1.401-.109c.429 0 .834.053 1.217.157s.718.271 1.005.499.515.52.684.875.253.774.253 1.258c0 .282-.05.54-.15.772s-.228.436-.383.608-.328.316-.52.431-.367.191-.531.227z"/></g></svg>',
				'audio'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#7d6599"/><path d="M35.67 14.986c-.567-.796-1.3-1.543-2.308-2.351-3.914-3.131-4.757-6.277-4.862-6.738V5a1 1 0 10-2 0v18.412h-3.706c-3.882 0-6.294 1.961-6.294 5.117 0 3.466 2.24 5.706 5.706 5.706a6.3 6.3 0 006.294-6.294V16.468l.298.243c.34.336.861.72 1.521 1.205 2.318 1.709 6.2 4.567 5.224 7.793a1 1 0 101.914.581c1.627-5.375-.422-9.39-1.787-11.304zM26.5 27.941a4.299 4.299 0 01-4.294 4.294c-2.355 0-3.706-1.351-3.706-3.706 0-2.576 2.335-3.117 4.294-3.117H26.5v2.529zm5.005-11.633c-.571-.422-1.065-.785-1.371-1.081l-1.634-1.34v-3.473c.827 1.174 1.987 2.483 3.612 3.783.858.688 1.472 1.308 1.929 1.95a10.832 10.832 0 011.788 3.978c-1.327-1.61-3.084-2.904-4.324-3.817z" fill="#c8bdb8"/><g fill="#fff"><path d="M22.4 42.924h1.668V53H22.4v-6.932l-2.256 5.605h-1.449l-2.27-5.605V53h-1.668V42.924h1.668l2.994 6.891 2.981-6.891zM28.211 53H26.57V42.924h2.898c.429 0 .853.068 1.271.205s.795.342 1.128.615.602.604.807.991.308.822.308 1.306c0 .511-.087.973-.26 1.388s-.415.764-.725 1.046-.684.501-1.121.656-.921.232-1.449.232H28.21V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103s.376-.18.54-.335.296-.371.396-.649.15-.622.15-1.032c0-.164-.022-.354-.068-.567s-.139-.419-.28-.615-.34-.36-.595-.492-.593-.198-1.012-.198h-1.23zM38.615 47.695a2.2 2.2 0 011.271.67c.191.2.351.444.479.731s.191.617.191.991c0 .456-.066.85-.198 1.183s-.299.608-.499.827-.426.395-.677.526-.494.23-.731.294-.453.104-.649.123-.349.027-.458.027c-.766 0-1.369-.053-1.812-.157s-.75-.212-.923-.321l.369-1.176c.082.046.159.096.232.15s.178.107.314.157.328.091.574.123.583.048 1.012.048c.629 0 1.099-.171 1.408-.513s.465-.772.465-1.292c0-.492-.142-.907-.424-1.244s-.697-.506-1.244-.506H35.92h.014l-.014-1.107h.93c.2 0 .403-.005.608-.014s.398-.06.581-.15.333-.246.451-.465.178-.533.178-.943c0-.164-.014-.337-.041-.52s-.103-.351-.226-.506-.303-.28-.54-.376-.565-.144-.984-.144-.754.039-1.005.116-.439.139-.567.185l-.479-1.23c.219-.063.49-.132.813-.205s.791-.109 1.401-.109c.429 0 .834.053 1.217.157s.718.271 1.005.499.515.52.684.875.253.774.253 1.258c0 .282-.05.54-.15.772s-.228.436-.383.608-.328.316-.52.431-.367.191-.531.227z"/></g></svg>',
				// image files
				'jpg'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><circle cx="18.931" cy="14.431" r="4.569" fill="#f3d55b"/><path fill="#26b99a" d="M6.5 39h43V28l-10-9.5L29 30l-5.483-5.483z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#14a085"/><g fill="#fff"><path d="M21.426 42.65v7.848c0 .474-.087.873-.26 1.196a2.142 2.142 0 01-.697.779 2.778 2.778 0 01-1.005.41 5.78 5.78 0 01-1.169.117c-.201 0-.436-.021-.704-.062-.269-.041-.547-.104-.834-.191s-.563-.185-.827-.294a3.022 3.022 0 01-.67-.369l.697-1.107c.091.063.221.13.39.198.168.068.353.132.554.191.2.06.41.111.629.157s.424.068.615.068c.483 0 .868-.094 1.155-.28s.439-.504.458-.95V42.65h1.668zM25.514 52.932h-1.641V42.855h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232h-1.217v3.638zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.256-.133-.593-.199-1.012-.199h-1.23zM39.5 47.736v3.896c-.21.265-.444.48-.704.649s-.533.308-.82.417-.583.187-.889.232c-.306.048-.608.07-.909.07-.602 0-1.155-.109-1.661-.328s-.948-.542-1.326-.971c-.378-.429-.675-.966-.889-1.613-.214-.647-.321-1.395-.321-2.242s.107-1.593.321-2.235c.214-.643.51-1.178.889-1.606a3.78 3.78 0 011.333-.978 4.069 4.069 0 011.654-.335c.547 0 1.057.091 1.531.273.474.183.897.456 1.271.82l-1.135 1.012a1.951 1.951 0 00-.752-.574 2.252 2.252 0 00-.875-.178c-.337 0-.659.063-.964.191a2.065 2.065 0 00-.82.649c-.242.306-.431.699-.567 1.183s-.21 1.075-.219 1.777c.009.684.08 1.276.212 1.777.132.501.314.911.547 1.23s.497.556.793.711c.296.155.608.232.937.232.1 0 .234-.007.403-.021a4.91 4.91 0 00.506-.068 3.29 3.29 0 00.485-.13.695.695 0 00.342-.232v-2.488h-1.709v-1.121H39.5z"/></g></svg>',
				'png'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#659c35"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><g fill="#fff"><path d="M17.385 53h-1.641V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232h-1.217V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335.164-.155.296-.371.396-.649.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23zM31.316 42.924V53h-1.668l-3.951-6.945V53h-1.668V42.924h1.668l3.951 6.945v-6.945h1.668zM41.16 47.805v3.896c-.21.265-.444.48-.704.649s-.533.308-.82.417-.584.187-.889.233a6.096 6.096 0 01-.909.068c-.602 0-1.155-.109-1.661-.328s-.948-.542-1.326-.971c-.378-.429-.675-.966-.889-1.613-.214-.647-.321-1.395-.321-2.242s.107-1.593.321-2.235c.214-.643.51-1.178.889-1.606a3.78 3.78 0 011.333-.978 4.069 4.069 0 011.654-.335c.547 0 1.057.091 1.531.273.474.183.897.456 1.271.82l-1.135 1.012a1.951 1.951 0 00-.752-.574 2.252 2.252 0 00-.875-.178c-.337 0-.659.063-.964.191a2.065 2.065 0 00-.82.649c-.242.306-.431.699-.567 1.183s-.21 1.075-.219 1.777c.009.684.08 1.276.212 1.777.132.501.314.911.547 1.23s.497.556.793.711c.296.155.608.232.937.232.1 0 .234-.007.403-.021a4.91 4.91 0 00.506-.068 3.29 3.29 0 00.485-.13.695.695 0 00.342-.232v-2.488h-1.709v-1.121h3.336z"/></g><circle cx="18.931" cy="14.431" r="4.569" fill="#f3d55b"/><path fill="#88c057" d="M6.5 39h43V28l-10-9.5L29 30l-5.483-5.483z"/></svg>',
				'psd'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#5889c4"/><g fill="#fff"><path d="M17.385 53h-1.641V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232h-1.217V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335.164-.155.296-.371.396-.649.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23zM29.73 50.238c0 .364-.075.718-.226 1.06s-.362.643-.636.902-.611.467-1.012.622a3.771 3.771 0 01-1.367.232c-.219 0-.444-.012-.677-.034s-.467-.062-.704-.116a3.686 3.686 0 01-.677-.226 2.17 2.17 0 01-.554-.349l.287-1.176c.127.073.289.144.485.212s.398.132.608.191c.209.06.419.107.629.144.209.036.405.055.588.055.556 0 .982-.13 1.278-.39.296-.26.444-.645.444-1.155 0-.31-.105-.574-.314-.793-.21-.219-.472-.417-.786-.595s-.654-.355-1.019-.533a6.173 6.173 0 01-1.025-.629 3.281 3.281 0 01-.793-.854c-.21-.328-.314-.738-.314-1.23 0-.446.082-.843.246-1.189s.385-.641.663-.882.602-.426.971-.554.759-.191 1.169-.191c.419 0 .843.039 1.271.116.428.077.774.203 1.039.376a13.31 13.31 0 01-.191.39l-.205.396c-.064.123-.119.226-.164.308a1.263 1.263 0 01-.082.137c-.055-.027-.116-.063-.185-.109s-.167-.091-.294-.137a2.097 2.097 0 00-.506-.096 4.781 4.781 0 00-.807.014c-.183.019-.355.07-.52.157s-.31.193-.438.321a1.464 1.464 0 00-.301.431 1.1 1.1 0 00-.109.458c0 .364.104.658.314.882.209.224.469.419.779.588.31.169.647.333 1.012.492.364.159.704.354 1.019.581s.576.513.786.854c.213.342.318.781.318 1.319zM39.629 47.682c0 .829-.089 1.538-.267 2.126s-.403 1.08-.677 1.477-.581.709-.923.937-.672.398-.991.513a4.094 4.094 0 01-.875.219c-.264.03-.46.046-.587.046h-3.814V42.924h3.035c.848 0 1.593.135 2.235.403s1.176.627 1.6 1.073.74.955.95 1.524c.209.57.314 1.156.314 1.758zm-4.867 4.115c1.112 0 1.914-.355 2.406-1.066s.738-1.741.738-3.09c0-.419-.05-.834-.15-1.244-.101-.41-.294-.781-.581-1.114s-.677-.602-1.169-.807-1.13-.308-1.914-.308h-.957v7.629h1.627z"/></g><path d="M38.894 11.163a1.001 1.001 0 00-1.384.291c-1.433 2.196-3.979 5.164-5.19 6.375l-2.788 2.788-2.828-2.828L29.49 15c1.771-1.771 6.271-5.442 8.116-6.102.04-.006.091-.015.112-.02.521-.117.827-.617.735-1.143-.093-.526-.616-.885-1.143-.814l-.283.062c-2.439.813-7.457 5.108-8.952 6.604l-7.185 7.185c-.095-.004-.189-.014-.285-.014a6.072 6.072 0 00-5.983 5.016l-.108.073-.02.536c-.034.942-.302 4.052-2.343 4.546a1.057 1.057 0 00-.793 1.152l.162.919h7l2.086-.03a6.114 6.114 0 006.107-6.107c0-.193-.011-.383-.029-.572l7.049-7.048c1.305-1.305 3.925-4.358 5.451-6.696a1 1 0 00-.29-1.384zM20.606 30.97h-3.739l.015.03h-1.661c.887-1.341 1.161-3.037 1.244-4.045l.082-.504a4.08 4.08 0 014.06-3.694c.121 0 .235.025.354.036l.167.021a4.09 4.09 0 012.6 1.411c.058.069.118.135.171.208a4.176 4.176 0 01.368.605c.063.126.124.254.174.387.03.08.052.164.077.247.045.147.091.294.12.449l.005.026c.041.234.072.471.072.716a4.115 4.115 0 01-4.109 4.107zm4.91-7.723l-.02-.026a6.128 6.128 0 00-2.332-1.895l2.123-2.123 2.828 2.828-2.071 2.07a5.893 5.893 0 00-.528-.854z" fill="#c8bdb8"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/></svg>',
				'image'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><circle cx="18.931" cy="14.431" r="4.569" fill="#f3d55b"/><path fill="#26b99a" d="M6.5 39h43V28l-10-9.5L29 30l-5.483-5.483z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#14a085"/><g fill="#fff"><path d="M21.426 42.65v7.848c0 .474-.087.873-.26 1.196a2.142 2.142 0 01-.697.779 2.778 2.778 0 01-1.005.41 5.78 5.78 0 01-1.169.117c-.201 0-.436-.021-.704-.062-.269-.041-.547-.104-.834-.191s-.563-.185-.827-.294a3.022 3.022 0 01-.67-.369l.697-1.107c.091.063.221.13.39.198.168.068.353.132.554.191.2.06.41.111.629.157s.424.068.615.068c.483 0 .868-.094 1.155-.28s.439-.504.458-.95V42.65h1.668zM25.514 52.932h-1.641V42.855h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232h-1.217v3.638zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.256-.133-.593-.199-1.012-.199h-1.23zM39.5 47.736v3.896c-.21.265-.444.48-.704.649s-.533.308-.82.417-.583.187-.889.232c-.306.048-.608.07-.909.07-.602 0-1.155-.109-1.661-.328s-.948-.542-1.326-.971c-.378-.429-.675-.966-.889-1.613-.214-.647-.321-1.395-.321-2.242s.107-1.593.321-2.235c.214-.643.51-1.178.889-1.606a3.78 3.78 0 011.333-.978 4.069 4.069 0 011.654-.335c.547 0 1.057.091 1.531.273.474.183.897.456 1.271.82l-1.135 1.012a1.951 1.951 0 00-.752-.574 2.252 2.252 0 00-.875-.178c-.337 0-.659.063-.964.191a2.065 2.065 0 00-.82.649c-.242.306-.431.699-.567 1.183s-.21 1.075-.219 1.777c.009.684.08 1.276.212 1.777.132.501.314.911.547 1.23s.497.556.793.711c.296.155.608.232.937.232.1 0 .234-.007.403-.021a4.91 4.91 0 00.506-.068 3.29 3.29 0 00.485-.13.695.695 0 00.342-.232v-2.488h-1.709v-1.121H39.5z"/></g></svg>',
				// video files
				'avi'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#d75e72"/><g fill="#fff"><path d="M16.361 53l3.527-10.063h1.627L25.016 53H23.32l-.725-2.092h-3.787L18.07 53h-1.709zm2.83-3.199h3.008l-1.49-4.348-1.518 4.348zM29.828 53.055l-3.131-10.131h1.873l2.338 8.695 2.475-8.695h1.859l-3.281 10.131h-2.133zM38.988 53H37.32V42.924h1.668V53z"/></g><path fill="#c8bdb8" d="M23.5 28V14l11 7z"/></svg>',
				'mp4'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#ff5364"/><g fill="#fff"><path d="M22.4 42.924h1.668V53H22.4v-6.932l-2.256 5.605h-1.449l-2.27-5.605V53h-1.668V42.924h1.668l2.994 6.891 2.981-6.891zM28.211 53H26.57V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232H28.21V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23zM38.479 50.648h-4.361V49.35l4.361-6.426h1.668v6.426H41.2v1.299h-1.053V53h-1.668v-2.352zm0-1.298v-4.512L35.58 49.35h2.899z"/></g><path d="M24.5 28a1 1 0 01-1-1V13a1.001 1.001 0 011.538-.844l11 7a1.003 1.003 0 01-.001 1.688l-11 7c-.163.103-.35.156-.537.156zm1-13.179v10.357L33.637 20 25.5 14.821z" fill="#c8bdb8"/><path d="M28.5 35c-8.271 0-15-6.729-15-15s6.729-15 15-15 15 6.729 15 15-6.729 15-15 15zm0-28c-7.168 0-13 5.832-13 13s5.832 13 13 13 13-5.832 13-13-5.832-13-13-13z" fill="#c8bdb8"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/></svg>',
				'video'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#ff5364"/><g fill="#fff"><path d="M22.4 42.924h1.668V53H22.4v-6.932l-2.256 5.605h-1.449l-2.27-5.605V53h-1.668V42.924h1.668l2.994 6.891 2.981-6.891zM28.211 53H26.57V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615.333.273.602.604.807.991s.308.822.308 1.306c0 .511-.087.973-.26 1.388a2.9 2.9 0 01-.725 1.046c-.31.282-.684.501-1.121.656s-.921.232-1.449.232H28.21V53zm0-8.832v3.992h1.504c.2 0 .398-.034.595-.103.196-.068.376-.18.54-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567a1.637 1.637 0 00-.28-.615 1.657 1.657 0 00-.595-.492c-.255-.132-.593-.198-1.012-.198h-1.23zM38.479 50.648h-4.361V49.35l4.361-6.426h1.668v6.426H41.2v1.299h-1.053V53h-1.668v-2.352zm0-1.298v-4.512L35.58 49.35h2.899z"/></g><path d="M24.5 28a1 1 0 01-1-1V13a1.001 1.001 0 011.538-.844l11 7a1.003 1.003 0 01-.001 1.688l-11 7c-.163.103-.35.156-.537.156zm1-13.179v10.357L33.637 20 25.5 14.821z" fill="#c8bdb8"/><path d="M28.5 35c-8.271 0-15-6.729-15-15s6.729-15 15-15 15 6.729 15 15-6.729 15-15 15zm0-28c-7.168 0-13 5.832-13 13s5.832 13 13 13 13-5.832 13-13-5.832-13-13-13z" fill="#c8bdb8"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/></svg>',
				// misc files
				'csv'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#f36fa0"/><g fill="#fff"><path d="M21.58 51.975a3.73 3.73 0 01-1.271.82 4.212 4.212 0 01-1.531.273c-.602 0-1.155-.109-1.661-.328s-.948-.542-1.326-.971c-.378-.429-.675-.966-.889-1.613-.214-.647-.321-1.395-.321-2.242s.107-1.593.321-2.235c.214-.643.51-1.178.889-1.606a3.78 3.78 0 011.333-.978 4.069 4.069 0 011.654-.335c.547 0 1.057.091 1.531.273.474.183.897.456 1.271.82l-1.135 1.012a2.072 2.072 0 00-1.627-.752c-.337 0-.659.063-.964.191a2.065 2.065 0 00-.82.649c-.242.306-.431.699-.567 1.183s-.21 1.075-.219 1.777c.009.684.08 1.267.212 1.75.132.483.314.877.547 1.183s.497.528.793.67c.296.142.608.212.937.212s.636-.06.923-.178.549-.31.786-.574l1.134.999zM29.633 50.238c0 .364-.075.718-.226 1.06s-.362.643-.636.902-.611.467-1.012.622a3.771 3.771 0 01-1.367.232c-.219 0-.444-.012-.677-.034s-.467-.062-.704-.116a3.686 3.686 0 01-.677-.226 2.17 2.17 0 01-.554-.349l.287-1.176c.127.073.289.144.485.212s.398.132.608.191c.209.06.419.107.629.144.209.036.405.055.588.055.556 0 .982-.13 1.278-.39.296-.26.444-.645.444-1.155 0-.31-.105-.574-.314-.793-.21-.219-.472-.417-.786-.595s-.654-.355-1.019-.533a6.173 6.173 0 01-1.025-.629 3.281 3.281 0 01-.793-.854c-.21-.328-.314-.738-.314-1.23 0-.446.082-.843.246-1.189s.385-.641.663-.882.602-.426.971-.554.759-.191 1.169-.191c.419 0 .843.039 1.271.116.428.077.774.203 1.039.376a13.31 13.31 0 01-.191.39l-.205.396c-.064.123-.119.226-.164.308a1.263 1.263 0 01-.082.137c-.055-.027-.116-.063-.185-.109s-.167-.091-.294-.137a2.097 2.097 0 00-.506-.096 4.781 4.781 0 00-.807.014c-.183.019-.355.07-.52.157s-.31.193-.438.321a1.464 1.464 0 00-.301.431 1.1 1.1 0 00-.109.458c0 .364.104.658.314.882.209.224.469.419.779.588.31.169.647.333 1.012.492.364.159.704.354 1.019.581s.576.513.786.854c.213.342.318.781.318 1.319zM34.035 53.055l-3.131-10.131h1.873l2.338 8.695 2.475-8.695h1.859l-3.281 10.131h-2.133z"/></g><path d="M23.5 16v-4h-12v22h33V16h-21zm-10-2h8v2h-8v-2zm0 4h8v2h-8v-2zm0 4h8v2h-8v-2zm0 4h8v2h-8v-2zm8 6h-8v-2h8v2zm21 0h-19v-2h19v2zm0-4h-19v-2h19v2zm0-4h-19v-2h19v2zm-19-4v-2h19v2h-19z" fill="#c8bdb8"/></svg>',
				'txt'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#95a5a5"/><g fill="#fff"><path d="M21.867 42.924v1.121h-3.008V53h-1.654v-8.955h-3.008v-1.121h7.67zM28.443 48.105L31 53h-1.9l-1.6-3.801h-.137L25.641 53h-1.9l2.557-4.895-2.721-5.182h1.873l1.777 4.102h.137l1.928-4.102h1.873l-2.722 5.182zM40.529 42.924v1.121h-3.008V53h-1.654v-8.955h-3.008v-1.121h7.67z"/></g><path d="M18.5 13h-6a1 1 0 110-2h6a1 1 0 110 2zM21.5 18h-9a1 1 0 110-2h9a1 1 0 110 2zM25.5 18c-.26 0-.521-.11-.71-.29-.181-.19-.29-.44-.29-.71s.109-.52.3-.71c.36-.37 1.04-.37 1.41 0 .18.19.29.45.29.71 0 .26-.11.52-.29.71-.19.18-.45.29-.71.29zM37.5 18h-8a1 1 0 110-2h8a1 1 0 110 2zM12.5 33c-.26 0-.521-.11-.71-.29-.181-.19-.29-.45-.29-.71 0-.26.109-.52.29-.71.37-.37 1.05-.37 1.42.01.18.18.29.44.29.7 0 .26-.11.52-.29.71-.19.18-.45.29-.71.29zM24.5 33h-8a1 1 0 110-2h8a1 1 0 110 2zM43.5 18h-2a1 1 0 110-2h2a1 1 0 110 2zM34.5 23h-22a1 1 0 110-2h22a1 1 0 110 2zM43.5 23h-6a1 1 0 110-2h6a1 1 0 110 2zM16.5 28h-4a1 1 0 110-2h4a1 1 0 110 2zM30.5 28h-10a1 1 0 110-2h10a1 1 0 110 2zM43.5 28h-9a1 1 0 110-2h9a1 1 0 110 2z" fill="#c8bdb8"/></svg>',
				'rtf'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#90bae1"/><g fill="#fff"><path d="M20.83 53l-1.9-3.65c-.027 0-.064.002-.109.007-.046.005-.116.007-.212.007h-1.238V53H15.73V42.924h2.898c.428 0 .852.068 1.271.205.419.137.795.342 1.128.615s.602.604.807.991.308.822.308 1.306c0 .729-.169 1.349-.506 1.859a3.034 3.034 0 01-1.367 1.135L22.566 53H20.83zm-3.459-4.84h1.49c.2 0 .4-.034.602-.103.2-.068.383-.18.547-.335s.296-.371.396-.649c.1-.278.15-.622.15-1.032 0-.164-.023-.354-.068-.567-.046-.214-.139-.419-.28-.615s-.34-.36-.595-.492c-.256-.132-.593-.198-1.012-.198h-1.23v3.991zM31.891 42.924v1.121h-3.008V53h-1.654v-8.955h-3.008v-1.121h7.67zM35.555 44.168v3.172h4.211v1.121h-4.211V53h-1.668V42.924h6.303v1.244h-4.635z"/></g><g fill="#c8bdb8"><path d="M12.5 13h6a1 1 0 100-2h-6a1 1 0 100 2zM12.5 18h9a1 1 0 100-2h-9a1 1 0 100 2zM25.5 18c.26 0 .52-.11.71-.29.18-.19.29-.45.29-.71 0-.26-.11-.52-.29-.71-.38-.37-1.04-.37-1.42 0-.181.19-.29.44-.29.71s.109.52.29.71c.189.18.45.29.71.29zM29.5 18h8a1 1 0 100-2h-8a1 1 0 100 2zM11.79 31.29c-.181.19-.29.44-.29.71s.109.52.29.71c.189.18.439.29.71.29.27 0 .52-.11.71-.29.18-.19.29-.45.29-.71 0-.26-.11-.52-.29-.71-.37-.37-1.05-.37-1.42 0zM24.5 31h-8a1 1 0 100 2h8a1 1 0 100-2zM41.5 18h2a1 1 0 100-2h-2a1 1 0 100 2zM12.5 23h22a1 1 0 100-2h-22a1 1 0 100 2zM43.5 21h-6a1 1 0 100 2h6a1 1 0 100-2zM12.5 28h4a1 1 0 100-2h-4a1 1 0 100 2zM30.5 26h-10a1 1 0 100 2h10a1 1 0 100-2zM43.5 26h-9a1 1 0 100 2h9a1 1 0 100-2z"/></g></svg>',
				'css'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#0096e6"/><g fill="#fff"><path d="M23.58 51.975c-.374.364-.798.638-1.271.82s-.984.273-1.531.273c-.602 0-1.155-.109-1.661-.328s-.948-.542-1.326-.971-.675-.966-.889-1.613c-.214-.647-.321-1.395-.321-2.242s.107-1.593.321-2.235c.214-.643.511-1.178.889-1.606s.822-.754 1.333-.978 1.062-.335 1.654-.335c.547 0 1.058.091 1.531.273s.897.456 1.271.82l-1.135 1.012c-.228-.265-.48-.456-.759-.574s-.567-.178-.868-.178c-.337 0-.658.063-.964.191s-.579.344-.82.649-.431.699-.567 1.183-.21 1.075-.219 1.777c.009.684.08 1.267.212 1.75s.314.877.547 1.183.497.528.793.67.608.212.937.212.636-.06.923-.178.549-.31.786-.574l1.134.999zM31.633 50.238c0 .364-.075.718-.226 1.06s-.362.643-.636.902-.61.467-1.012.622-.856.232-1.367.232c-.219 0-.444-.012-.677-.034s-.467-.062-.704-.116-.463-.13-.677-.226-.398-.212-.554-.349l.287-1.176c.128.073.289.144.485.212s.398.132.608.191.419.107.629.144.405.055.588.055c.556 0 .982-.13 1.278-.39s.444-.645.444-1.155c0-.31-.104-.574-.314-.793s-.472-.417-.786-.595-.654-.355-1.019-.533-.706-.388-1.025-.629-.583-.526-.793-.854-.314-.738-.314-1.23c0-.446.082-.843.246-1.189s.385-.641.663-.882.602-.426.971-.554.759-.191 1.169-.191c.419 0 .843.039 1.271.116s.774.203 1.039.376c-.055.118-.118.248-.191.39l-.205.396c-.063.123-.118.226-.164.308s-.073.128-.082.137c-.055-.027-.116-.063-.185-.109s-.166-.091-.294-.137-.296-.077-.506-.096-.479-.014-.807.014c-.183.019-.355.07-.52.157s-.31.193-.438.321-.228.271-.301.431-.109.313-.109.458c0 .364.104.658.314.882s.47.419.779.588.647.333 1.012.492.704.354 1.019.581.576.513.786.854.318.781.318 1.319zM39.043 50.238c0 .364-.075.718-.226 1.06s-.362.643-.636.902-.61.467-1.012.622-.856.232-1.367.232c-.219 0-.444-.012-.677-.034s-.467-.062-.704-.116-.463-.13-.677-.226-.398-.212-.554-.349l.287-1.176c.128.073.289.144.485.212s.398.132.608.191.419.107.629.144.405.055.588.055c.556 0 .982-.13 1.278-.39s.444-.645.444-1.155c0-.31-.104-.574-.314-.793s-.472-.417-.786-.595-.654-.355-1.019-.533-.706-.388-1.025-.629-.583-.526-.793-.854-.314-.738-.314-1.23c0-.446.082-.843.246-1.189s.385-.641.663-.882.602-.426.971-.554.759-.191 1.169-.191c.419 0 .843.039 1.271.116s.774.203 1.039.376c-.055.118-.118.248-.191.39l-.205.396c-.063.123-.118.226-.164.308s-.073.128-.082.137c-.055-.027-.116-.063-.185-.109s-.166-.091-.294-.137-.296-.077-.506-.096-.479-.014-.807.014c-.183.019-.355.07-.52.157s-.31.193-.438.321-.228.271-.301.431-.109.313-.109.458c0 .364.104.658.314.882s.47.419.779.588.647.333 1.012.492.704.354 1.019.581.576.513.786.854.318.781.318 1.319z"/></g><g fill="#0096e6"><path d="M19.5 19v-4c0-.551.448-1 1-1a1 1 0 100-2c-1.654 0-3 1.346-3 3v4c0 1.103-.897 2-2 2a1 1 0 100 2c1.103 0 2 .897 2 2v4c0 1.654 1.346 3 3 3a1 1 0 100-2c-.552 0-1-.449-1-1v-4c0-1.2-.542-2.266-1.382-3a3.975 3.975 0 001.382-3zM39.5 21c-1.103 0-2-.897-2-2v-4c0-1.654-1.346-3-3-3a1 1 0 100 2c.552 0 1 .449 1 1v4c0 1.2.542 2.266 1.382 3a3.975 3.975 0 00-1.382 3v4c0 .551-.448 1-1 1a1 1 0 100 2c1.654 0 3-1.346 3-3v-4c0-1.103.897-2 2-2a1 1 0 100-2z"/></g></svg>',
				'html'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#ec6630"/><g fill="#fff"><path d="M17.455 42.924V53h-1.641v-4.539h-4.361V53H9.785V42.924h1.668v4.416h4.361v-4.416h1.641zM27.107 42.924v1.121H24.1V53h-1.654v-8.955h-3.008v-1.121h7.669zM36.705 42.924h1.668V53h-1.668v-6.932l-2.256 5.605H33l-2.27-5.605V53h-1.668V42.924h1.668l2.994 6.891 2.981-6.891zM42.57 42.924v8.832h4.635V53h-6.303V42.924h1.668z"/></g><g fill="#ec6630"><path d="M23.207 16.293a.999.999 0 00-1.414 0l-6 6a.999.999 0 000 1.414l6 6a.997.997 0 001.414 0 .999.999 0 000-1.414L17.914 23l5.293-5.293a.999.999 0 000-1.414zM41.207 22.293l-6-6a.999.999 0 10-1.414 1.414L39.086 23l-5.293 5.293a.999.999 0 101.414 1.414l6-6a.999.999 0 000-1.414z"/></g></svg>',
				'js'           => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 56 56"><path d="M36.985 0H7.963C7.155 0 6.5.655 6.5 1.926V55c0 .345.655 1 1.463 1h40.074c.808 0 1.463-.655 1.463-1V12.978c0-.696-.093-.92-.257-1.085L37.607.257A.884.884 0 0036.985 0z" fill="#e9e9e0"/><path fill="#d9d7ca" d="M37.5.151V12h11.849z"/><path d="M48.037 56H7.963A1.463 1.463 0 016.5 54.537V39h43v15.537c0 .808-.655 1.463-1.463 1.463z" fill="#eeaf4b"/><g fill="#fff"><path d="M26.021 42.719v7.848c0 .474-.087.873-.26 1.196a2.151 2.151 0 01-.697.779c-.292.196-.627.333-1.005.41s-.769.116-1.169.116c-.201 0-.436-.021-.704-.062s-.547-.104-.834-.191-.563-.185-.827-.294a3.022 3.022 0 01-.67-.369l.697-1.107c.091.063.221.13.39.198s.353.132.554.191c.2.06.41.111.629.157s.424.068.615.068c.482 0 .868-.094 1.155-.28s.439-.504.458-.95v-7.711h1.668zM34.184 50.238c0 .364-.075.718-.226 1.06s-.362.643-.636.902-.611.467-1.012.622a3.771 3.771 0 01-1.367.232c-.219 0-.444-.012-.677-.034s-.468-.062-.704-.116c-.237-.055-.463-.13-.677-.226s-.399-.212-.554-.349l.287-1.176c.127.073.289.144.485.212s.398.132.608.191c.209.06.419.107.629.144.209.036.405.055.588.055.556 0 .982-.13 1.278-.39s.444-.645.444-1.155c0-.31-.105-.574-.314-.793-.21-.219-.472-.417-.786-.595s-.654-.355-1.019-.533a6.173 6.173 0 01-1.025-.629 3.264 3.264 0 01-.793-.854c-.21-.328-.314-.738-.314-1.23 0-.446.082-.843.246-1.189s.385-.641.663-.882.602-.426.971-.554.759-.191 1.169-.191c.419 0 .843.039 1.271.116.428.077.774.203 1.039.376a13.31 13.31 0 01-.191.39l-.205.396c-.064.123-.119.226-.164.308a1.263 1.263 0 01-.082.137c-.055-.027-.116-.063-.185-.109s-.167-.091-.294-.137a2.107 2.107 0 00-.506-.096 4.781 4.781 0 00-.807.014c-.183.019-.355.07-.52.157s-.311.193-.438.321a1.446 1.446 0 00-.301.431 1.1 1.1 0 00-.109.458c0 .364.104.658.314.882.209.224.469.419.779.588.31.169.646.333 1.012.492.364.159.704.354 1.019.581s.576.513.786.854c.212.342.318.781.318 1.319z"/></g><g fill="#eeaf4b"><path d="M19.5 19v-4c0-.551.448-1 1-1a1 1 0 100-2c-1.654 0-3 1.346-3 3v4c0 1.103-.897 2-2 2a1 1 0 100 2c1.103 0 2 .897 2 2v4c0 1.654 1.346 3 3 3a1 1 0 100-2c-.552 0-1-.449-1-1v-4c0-1.2-.542-2.266-1.382-3a3.975 3.975 0 001.382-3z"/><circle cx="27.5" cy="18.5" r="1.5"/><path d="M39.5 21c-1.103 0-2-.897-2-2v-4c0-1.654-1.346-3-3-3a1 1 0 100 2c.552 0 1 .449 1 1v4c0 1.2.542 2.266 1.382 3a3.975 3.975 0 00-1.382 3v4c0 .551-.448 1-1 1a1 1 0 100 2c1.654 0 3-1.346 3-3v-4c0-1.103.897-2 2-2a1 1 0 100-2zM27.5 24a1 1 0 00-1 1v3a1 1 0 102 0v-3a1 1 0 00-1-1z"/></g></svg>',
				// dlp items
				'folder'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M464 128H272l-64-64H48C21.49 64 0 85.49 0 112v288c0 26.51 21.49 48 48 48h416c26.51 0 48-21.49 48-48V176c0-26.51-21.49-48-48-48z"/></svg>',
				'folder_open'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M572.694 292.093L500.27 416.248A63.997 63.997 0 0 1 444.989 448H45.025c-18.523 0-30.064-20.093-20.731-36.093l72.424-124.155A64 64 0 0 1 152 256h399.964c18.523 0 30.064 20.093 20.73 36.093zM152 224h328v-48c0-26.51-21.49-48-48-48H272l-64-64H48C21.49 64 0 85.49 0 112v278.046l69.077-118.418C86.214 242.25 117.989 224 152 224z"/></svg>',
				'loading'      => '<svg viewBox="0 0 38 38" xmlns="http://www.w3.org/2000/svg" stroke="#000"><g fill="none" fill-rule="evenodd"><g transform="translate(1 1)" stroke-width="2"><circle stroke-opacity=".5" cx="18" cy="18" r="18"/><path d="M36 18c0-9.94-8.06-18-18-18"><animateTransform attributeName="transform" type="rotate" from="0 18 18" to="360 18 18" dur="1s" repeatCount="indefinite"/></path></g></g></svg>',
				'link'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" aria-label="' . esc_attr__( 'Link', 'document-library-pro' ) . '"><path fill="currentColor" d="M326.612 185.391c59.747 59.809 58.927 155.698.36 214.59-.11.12-.24.25-.36.37l-67.2 67.2c-59.27 59.27-155.699 59.262-214.96 0-59.27-59.26-59.27-155.7 0-214.96l37.106-37.106c9.84-9.84 26.786-3.3 27.294 10.606.648 17.722 3.826 35.527 9.69 52.721 1.986 5.822.567 12.262-3.783 16.612l-13.087 13.087c-28.026 28.026-28.905 73.66-1.155 101.96 28.024 28.579 74.086 28.749 102.325.51l67.2-67.19c28.191-28.191 28.073-73.757 0-101.83-3.701-3.694-7.429-6.564-10.341-8.569a16.037 16.037 0 0 1-6.947-12.606c-.396-10.567 3.348-21.456 11.698-29.806l21.054-21.055c5.521-5.521 14.182-6.199 20.584-1.731a152.482 152.482 0 0 1 20.522 17.197zM467.547 44.449c-59.261-59.262-155.69-59.27-214.96 0l-67.2 67.2c-.12.12-.25.25-.36.37-58.566 58.892-59.387 154.781.36 214.59a152.454 152.454 0 0 0 20.521 17.196c6.402 4.468 15.064 3.789 20.584-1.731l21.054-21.055c8.35-8.35 12.094-19.239 11.698-29.806a16.037 16.037 0 0 0-6.947-12.606c-2.912-2.005-6.64-4.875-10.341-8.569-28.073-28.073-28.191-73.639 0-101.83l67.2-67.19c28.239-28.239 74.3-28.069 102.325.51 27.75 28.3 26.872 73.934-1.155 101.96l-13.087 13.087c-4.35 4.35-5.769 10.79-3.783 16.612 5.864 17.194 9.042 34.999 9.69 52.721.509 13.906 17.454 20.446 27.294 10.606l37.106-37.106c59.271-59.259 59.271-155.699.001-214.959z"/></svg>',
				'download'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" aria-label="' . esc_attr__( 'Download', 'document-library-pro' ) . '"><path fill="currentColor" d="M216 0h80c13.3 0 24 10.7 24 24v168h87.7c17.8 0 26.7 21.5 14.1 34.1L269.7 378.3c-7.5 7.5-19.8 7.5-27.3 0L90.1 226.1c-12.6-12.6-3.7-34.1 14.1-34.1H192V24c0-13.3 10.7-24 24-24zm296 376v112c0 13.3-10.7 24-24 24H24c-13.3 0-24-10.7-24-24V376c0-13.3 10.7-24 24-24h146.7l49 49c20.1 20.1 52.5 20.1 72.6 0l49-49H488c13.3 0 24 10.7 24 24zm-124 88c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20zm64 0c0-11-9-20-20-20s-20 9-20 20 9 20 20 20 20-9 20-20z"></path></svg>',
				'preview'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" aria-label="' . esc_attr__( 'Preview', 'document-library-pro' ) . '"><path fill="currentColor" d="M505 442.7L405.3 343c-4.5-4.5-10.6-7-17-7H372c27.6-35.3 44-79.7 44-128C416 93.1 322.9 0 208 0S0 93.1 0 208s93.1 208 208 208c48.3 0 92.7-16.4 128-44v16.3c0 6.4 2.5 12.5 7 17l99.7 99.7c9.4 9.4 24.6 9.4 33.9 0l28.3-28.3c9.4-9.4 9.4-24.6.1-34zM208 336c-70.7 0-128-57.2-128-128 0-70.7 57.2-128 128-128 70.7 0 128 57.2 128 128 0 70.7-57.2 128-128 128z"></path></svg>',
				'close'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" aria-label="' . esc_attr__( 'Close', 'document-library-pro' ) . '"><path fill="currentColor" d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z" /><path d="M0 0h24v24H0z" fill="none" /></svg>',
				'reset'        => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" aria-label="' . esc_attr__( 'Reset', 'document-library-pro' ) . '"><path d="M16 2c-4.418 0-8.418 1.791-11.313 4.687l-4.686-4.687v12h12l-4.485-4.485c2.172-2.172 5.172-3.515 8.485-3.515 6.627 0 12 5.373 12 12 0 3.584-1.572 6.801-4.063 9l2.646 3c3.322-2.932 5.417-7.221 5.417-12 0-8.837-7.163-16-16-16z"></path></svg>',
			]
		);

		if ( isset( $icons[ $key ] ) ) {
			$classes = array_merge( [ 'dlp-icon', $key ], $classes );

			if ( $fill_color ) {
				$fill_color = sprintf( ' style="fill:%s;"', $fill_color );
			}

			return sprintf( '<div class="%1$s"%3$s>%2$s</div>', implode( ' ', $classes ), $icons[ $key ], $fill_color );
		}

		return false;
	}

	/**
	 * Sanitize the HTML markup of an SVG
	 *
	 * @param  string $svg The markup to be sanitized
	 * @return string      The sanitize markup
	 */
	public static function sanitize_svg( $svg ) {
		$sanitizer = new Sanitizer();

		return $sanitizer->sanitize( trim( $svg ) );
	}

	/**
	 * Get file extension icon.
	 *
	 * @param string $extension
	 * @return string
	 */
	public static function get_file_extension_icon( $extension ) {
		$extension = strtolower( $extension );

		$file_type_icon_map = apply_filters(
			'document_library_pro_file_type_icon_map',
			[
				// External
				'www'      => 'www',
				'external' => 'www',
				// Image formats.
				'jpg'      => 'jpg',
				'jpeg'     => 'jpg',
				'jpe'      => 'jpg',
				'gif'      => 'image',
				'png'      => 'png',
				'bmp'      => 'image',
				'tiff'     => 'image',
				'tif'      => 'image',
				'ico'      => 'image',
				'heic'     => 'image',
				'psd'      => 'psd',
				// Video formats.
				'asf'      => 'video',
				'asx'      => 'video',
				'wmv'      => 'video',
				'wmx'      => 'video',
				'wm'       => 'video',
				'avi'      => 'avi',
				'divx'     => 'video',
				'flv'      => 'video',
				'mov'      => 'video',
				'qt'       => 'video',
				'mpeg'     => 'mp4',
				'mpg'      => 'mp4',
				'mpe'      => 'mp4',
				'mp4'      => 'mp4',
				'm4v'      => 'video',
				'ogv'      => 'video',
				'webm'     => 'video',
				'mkv'      => 'video',
				'3gp'      => 'video', // Can also be audio.
				'3gpp'     => 'video', // Can also be audio.
				'3g2'      => 'video', // Can also be audio.
				'3gp2'     => 'video', // Can also be audio.
			// Text formats.
				'txt'      => 'txt',
				'asc'      => 'default',
				'c'        => 'default',
				'cc'       => 'default',
				'h'        => 'default',
				'srt'      => 'default',
				'css'      => 'css',
				'htm'      => 'html',
				'html'     => 'html',
				'vtt'      => 'default',
				'dfxp'     => 'default',
				'rtf'      => 'rtf',
				'js'       => 'js',
				'swf'      => 'default',
				'class'    => 'default',
				// Audio formats.
				'mp3'      => 'mp3',
				'm4a'      => 'audio',
				'm4b'      => 'audio',
				'aac'      => 'audio',
				'ra'       => 'audio',
				'ram'      => 'audio',
				'wav'      => 'audio',
				'ogg'      => 'audio',
				'oga'      => 'audio',
				'flac'     => 'audio',
				'mid'      => 'audio',
				'midi'     => 'audio',
				'wma'      => 'audio',
				'wax'      => 'audio',
				'mka'      => 'audio',
				// PDF
				'pdf'      => 'pdf',
				'xps'      => 'pdf',
				'oxps'     => 'pdf',
				// Calendar
				'ics'      => 'default',
				// Archives
				'tar'      => 'archive',
				'zip'      => 'archive',
				'gzip'     => 'archive',
				'gz'       => 'archive',
				'rar'      => 'archive',
				'7z'       => 'archive',
				'exe'      => 'exe',
				'xcf'      => 'archive',
				// Word Processor
				'doc'      => 'word',
				'wri'      => 'word',
				'docx'     => 'word',
				'docm'     => 'word',
				'dotx'     => 'word',
				'dotm'     => 'word',
				'wp'       => 'word',
				'wpd'      => 'word',
				'pages'    => 'word',
				'odt'      => 'word',
				'rtx'      => 'word',
				'onetoc'   => 'word',
				'onetoc2'  => 'word',
				'onetmp'   => 'word',
				'onepkg'   => 'word',
				// Presentation
				'pot'      => 'presentation',
				'pps'      => 'presentation',
				'ppt'      => 'presentation',
				'pptx'     => 'presentation',
				'pptm'     => 'presentation',
				'ppsx'     => 'presentation',
				'ppsm'     => 'presentation',
				'potx'     => 'presentation',
				'potm'     => 'presentation',
				'ppam'     => 'presentation',
				'sldx'     => 'presentation',
				'sldm'     => 'presentation',
				'key'      => 'presentation',
				'odp'      => 'presentation',
				'odc'      => 'presentation',
				'odg'      => 'presentation',
				// Spreadsheet
				'xla'      => 'spreadsheet',
				'xls'      => 'spreadsheet',
				'xlt'      => 'spreadsheet',
				'xlw'      => 'spreadsheet',
				'xlsx'     => 'spreadsheet',
				'xlsm'     => 'spreadsheet',
				'xlsb'     => 'spreadsheet',
				'xltx'     => 'spreadsheet',
				'xltm'     => 'spreadsheet',
				'xlam'     => 'spreadsheet',
				'numbers'  => 'spreadsheet',
				'ods'      => 'spreadsheet',
				'odb'      => 'spreadsheet',
				'odf'      => 'spreadsheet',
				'mdb'      => 'spreadsheet',
				'mpp'      => 'spreadsheet',
				'csv'      => 'csv',
				'tsv'      => 'spreadsheet',
			]
		);

		return isset( $file_type_icon_map[ $extension ] ) ? $file_type_icon_map[ $extension ] : 'default';
	}

}
